"""
================================================================================
 Script: summarize_archive.py
 Created at the request of the user by ChatGPT (GPT-4o)
 Date: 2025-12-12

 Description:
 This Python script was written upon the user's command to summarize the contents
 of an entire archive (folder and its subfolders) using AI heuristics and the 
 GPT-4o Mini model from OpenAI. It reads an API key from a text file one directory
 above the script (`../OpenAIKey.txt`), prompts the user for presentation metrics,
 and performs an intelligent analysis of each text-based file in the archive.

 Key Features:
 - Recursively collects readable text/code files (ignoring binaries, images, etc.)
 - Samples large files (start, middle, end) to stay within token limits
 - Uses OpenAI GPT-4o to summarize each file with user-defined goals
 - Compiles the output into a clean, professional PDF (`Summary.pdf`)
 - Asks user for title, focus/metrics, and optional keywords before analyzing
 - Designed for project review, documentation, technical summaries, or presentations

 Dependencies:
 - openai
 - reportlab

 Install required packages with:
     pip install openai reportlab

 Disclaimer:
 This script was written programmatically and can be extended as needed
 to support more complex file types, output formats, or summarization rules.
================================================================================
"""

import os
import openai
from reportlab.lib.pagesizes import LETTER
from reportlab.pdfgen import canvas
from reportlab.lib.units import inch
from datetime import datetime

# ---- Load API Key ----
def load_api_key(path="../OpenAIKey.txt"):
    try:
        with open(path, 'r') as f:
            return f.read().strip()
    except FileNotFoundError:
        print("OpenAIKey.txt not found. Ensure it's one directory up.")
        exit(1)

# ---- Prompt User for Summary Options ----
def prompt_user_inputs():
    print("== Archive Summarizer ==")
    title = input("Presentation Title: ")
    focus = input("Primary Focus or Metrics (comma-separated): ")
    keywords = input("Optional Keywords or Goals (comma-separated, leave blank if none): ")
    return title, focus, keywords

# ---- Collect Files Recursively ----
def collect_files(root="."):
    file_paths = []
    for dirpath, _, filenames in os.walk(root):
        for file in filenames:
            full_path = os.path.join(dirpath, file)
            if not file.lower().endswith((".png", ".jpg", ".jpeg", ".pdf", ".exe")):
                file_paths.append(full_path)
    return file_paths

# ---- Sample File Content ----
def sample_file_content(file_path, max_chars=2000):
    try:
        with open(file_path, 'r', encoding="utf-8", errors="ignore") as f:
            content = f.read()
            if len(content) > max_chars:
                # Take start, middle, end
                third = max_chars // 3
                return content[:third] + "\n...\n" + content[len(content)//2:len(content)//2 + third] + "\n...\n" + content[-third:]
            return content
    except Exception as e:
        return f"[Could not read file {file_path}: {e}]"

# ---- Analyze Content with OpenAI ----
def summarize_with_openai(file_contents, instructions, model="gpt-4o"):
    openai.api_key = load_api_key()
    try:
        response = openai.ChatCompletion.create(
            model=model,
            messages=[
                {"role": "system", "content": "You are an AI specialized in analyzing project archives and summarizing them for presentations."},
                {"role": "user", "content": instructions + "\n\n" + file_contents}
            ],
            temperature=0.5,
            max_tokens=1024
        )
        return response['choices'][0]['message']['content'].strip()
    except Exception as e:
        return f"[AI Error]: {e}"

# ---- Generate PDF Summary ----
def generate_pdf(title, focus, summaries, output_path="Summary.pdf"):
    c = canvas.Canvas(output_path, pagesize=LETTER)
    width, height = LETTER

    c.setFont("Helvetica-Bold", 18)
    c.drawString(1 * inch, height - 1 * inch, title)

    c.setFont("Helvetica", 12)
    c.drawString(1 * inch, height - 1.25 * inch, f"Focus: {focus}")
    c.drawString(1 * inch, height - 1.5 * inch, f"Generated: {datetime.now().strftime('%Y-%m-%d %H:%M')}")

    y = height - 2 * inch
    for path, summary in summaries.items():
        c.setFont("Helvetica-Bold", 11)
        c.drawString(1 * inch, y, f"{os.path.basename(path)}")
        y -= 0.2 * inch

        c.setFont("Helvetica", 10)
        for line in summary.split('\n'):
            if y < 1 * inch:
                c.showPage()
                y = height - 1 * inch
            c.drawString(1.2 * inch, y, line.strip())
            y -= 0.18 * inch
        y -= 0.1 * inch

    c.save()
    print(f"✅ Summary saved to {output_path}")

# ---- Main ----
def main():
    title, focus, keywords = prompt_user_inputs()
    files = collect_files()

    print(f"\nAnalyzing {len(files)} files...\n")
    summaries = {}
    for path in files:
        print(f"Processing: {path}")
        content = sample_file_content(path)
        prompt = f"Please summarize the following file as part of a presentation titled '{title}'. Focus on: {focus}."
        if keywords:
            prompt += f" Additional keywords or goals: {keywords}."
        summary = summarize_with_openai(content, prompt, model="gpt-4o")
        summaries[path] = summary

    generate_pdf(title, focus, summaries)

if __name__ == "__main__":
    main()
